// import { NextResponse } from "next/server";
// import crypto from "crypto";
// import { adminDb } from "@/lib/firebaseAdmin";
// import { requireFirebaseUser } from "@/lib/auth/requireIdToken";

// export async function POST(req: Request) {
//   try {
//     // 1) Check user (needs Firebase Admin + FIREBASE_SERVICE_ACCOUNT_KEY)
//     const user = await requireFirebaseUser(req as any);
//     if (!user.ok) {
//       return NextResponse.json({ ok: false, error: user.error }, { status: 401 });
//     }

//     // 2) Read Razorpay payload
//     const body = await req.json();
//     const { razorpay_payment_id, razorpay_order_id, razorpay_signature } = body;

//     if (!razorpay_payment_id || !razorpay_order_id || !razorpay_signature) {
//       return NextResponse.json(
//         { ok: false, error: "Missing payment verification fields" },
//         { status: 400 }
//       );
//     }

//     // 3) Verify signature
//     const secret = process.env.RAZORPAY_KEY_SECRET;
//     if (!secret) {
//       return NextResponse.json(
//         { ok: false, error: "Server misconfiguration: missing Razorpay secret" },
//         { status: 500 }
//       );
//     }

//     const payload = `${razorpay_order_id}|${razorpay_payment_id}`;

//     const expectedSignature = crypto
//       .createHmac("sha256", secret)
//       .update(payload)
//       .digest("hex");

//     if (expectedSignature !== razorpay_signature) {
//       return NextResponse.json(
//         { ok: false, error: "Invalid Razorpay signature" },
//         { status: 400 }
//       );
//     }

//     // 4) Mark user as paid in Firestore
//     const db = adminDb();
//     await db.collection("users").doc(user.uid).set(
//       {
//         hasFullAccess: true,
//         paidAt: new Date().toISOString(),
//         lastPayment: {
//           razorpay_payment_id,
//           razorpay_order_id,
//         },
//       },
//       { merge: true }
//     );

//     // ✅ Always return JSON
//     return NextResponse.json({ ok: true });
//   } catch (err: any) {
//     console.error("VERIFY PAYMENT ERROR:", err);
//     return NextResponse.json(
//       { ok: false, error: "Verification failed" },
//       { status: 500 }
//     );
//   }
// }

import { NextResponse } from "next/server";
import crypto from "crypto";
import { adminDb } from "@/lib/firebaseAdmin";
import { requireFirebaseUser } from "@/lib/auth/requireIdToken";

export async function POST(req: Request) {
  try {
    const user = await requireFirebaseUser(req as any);
    if (!user.ok) {
      return NextResponse.json({ ok: false, error: user.error }, { status: 401 });
    }

    const body = await req.json();
    const { razorpay_payment_id, razorpay_order_id, razorpay_signature, amount } = body;

    if (!razorpay_payment_id || !razorpay_order_id || !razorpay_signature) {
      return NextResponse.json(
        { ok: false, error: "Missing payment verification fields" },
        { status: 400 }
      );
    }

    const secret = process.env.RAZORPAY_KEY_SECRET;
    if (!secret) {
      return NextResponse.json(
        { ok: false, error: "Server misconfiguration: missing Razorpay secret" },
        { status: 500 }
      );
    }

    const payload = `${razorpay_order_id}|${razorpay_payment_id}`;
    const expectedSignature = crypto
      .createHmac("sha256", secret)
      .update(payload)
      .digest("hex");

    if (expectedSignature !== razorpay_signature) {
      return NextResponse.json(
        { ok: false, error: "Invalid Razorpay signature" },
        { status: 400 }
      );
    }

    const db = adminDb();

    // ✅ 1) Store payment record (payments collection)
    await db.collection("payments").doc(razorpay_payment_id).set(
      {
        uid: user.uid,
        orderId: razorpay_order_id,
        paymentId: razorpay_payment_id,
        signature: razorpay_signature, // optional; you can omit
        amount: typeof amount === "number" ? amount : null, // optional
        currency: "INR",
        status: "verified",
        paidAt: new Date().toISOString(),
      },
      { merge: true }
    );

    // ✅ 2) Update user access + link lastPaymentId
    await db.collection("users").doc(user.uid).set(
      {
        hasFullAccess: true,
        lastPaymentId: razorpay_payment_id,
        paidAt: new Date().toISOString(),
      },
      { merge: true }
    );

    // ✅ Return paymentId so UI can enable receipt immediately if needed
    return NextResponse.json({ ok: true, paymentId: razorpay_payment_id });
  } catch (err) {
    console.error("VERIFY PAYMENT ERROR:", err);
    return NextResponse.json(
      { ok: false, error: "Verification failed" },
      { status: 500 }
    );
  }
}
